<?php
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: login.php');
    exit;
}
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Accounts</h2>
    <div>
        <button id="refreshAccounts" class="btn btn-outline-primary btn-sm">
            <i class="bi bi-arrow-clockwise"></i> Refresh
        </button>
    </div>
    <div id="accounts-error" class="alert alert-danger d-none mt-3"></div>
}</div>

<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table align-middle" id="accountsTable">
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Type</th>
                        <th>Currency</th>
                        <th class="text-end">Balance</th>
                        <th style="width:140px;">Actions</th>
                    </tr>
                </thead>
                <tbody></tbody>
            </table>
        </div>
    </div>
    <div class="card-footer text-muted small">
        Balances reflect credits minus debits per account currency.
    </div>
    </div>

<!-- Statement Modal -->
<div class="modal fade" id="statementModal" tabindex="-1" aria-labelledby="statementModalLabel" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-scrollable">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="statementModalLabel">Account Statement</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form class="row g-2 mb-3" id="statementFilters">
            <input type="hidden" id="stmtAccountId" />
            <div class="col-md-3">
                <label class="form-label">From</label>
                <input type="date" id="stmtFrom" class="form-control" />
            </div>
            <div class="col-md-3">
                <label class="form-label">To</label>
                <input type="date" id="stmtTo" class="form-control" />
            </div>
            <div class="col-md-3">
                <label class="form-label">Currency</label>
                <input type="text" id="stmtCurrency" class="form-control" placeholder="e.g. USD"/>
            </div>
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-primary w-100"><i class="bi bi-funnel"></i> Apply</button>
            </div>
        </form>
        <div class="mb-2">
            <span class="badge bg-secondary me-2" id="openingBalance">Opening: 0.00</span>
            <span class="badge bg-secondary" id="closingBalance">Closing: 0.00</span>
        </div>
        <div class="table-responsive">
            <table class="table table-sm align-middle" id="statementTable">
                <thead>
                    <tr>
                        <th>Date</th>
                        <th>Type</th>
                        <th class="text-end">Amount</th>
                        <th>Currency</th>
                        <th>Description</th>
                        <th class="text-end">Running</th>
                    </tr>
                </thead>
                <tbody></tbody>
            </table>
        </div>
      </div>
    </div>
  </div>
  </div>

<script>
(function() {
    const $tableBody = $('#accountsTable tbody');
    const $error = $('#accounts-error');

    function loadAccounts() {
        $error.addClass('d-none').text('');
        $tableBody.html('<tr><td colspan="5" class="text-center text-muted">Loading...</td></tr>');
        fetch('api/accounts_get.php', { credentials: 'same-origin' })
            .then(r => r.json())
            .then(res => {
                if (!res.success) throw new Error(res.error || 'Failed to load');
                const rows = (res.data || []).map(acc => {
                    const currency = acc.currency || '—';
                    const balance = Number(acc.balance || 0).toFixed(2);
                    const type = (acc.type || '').toUpperCase();
                    const badgeClass = acc.is_active == 1 ? 'bg-success' : 'bg-secondary';
                    return `
                        <tr>
                            <td>
                                <span class="fw-semibold">${escapeHtml(acc.name)}</span>
                                <span class="badge ${badgeClass} ms-2">${acc.is_active == 1 ? 'Active' : 'Inactive'}</span>
                            </td>
                            <td>${escapeHtml(type)}</td>
                            <td>${escapeHtml(currency)}</td>
                            <td class="text-end">${balance}</td>
                            <td>
                                <button class="btn btn-sm btn-outline-primary" data-action="statement" data-id="${acc.id}" data-name="${escapeHtml(acc.name)}" data-currency="${escapeHtml(acc.currency || '')}">
                                    <i class="bi bi-list-check"></i> Statement
                                </button>
                            </td>
                        </tr>
                    `;
                }).join('');
                $tableBody.html(rows || '<tr><td colspan="5" class="text-center text-muted">No accounts found.</td></tr>');
            })
            .catch(err => {
                $error.removeClass('d-none').text(err.message);
                $tableBody.html('<tr><td colspan="5" class="text-center text-danger">Failed to load accounts</td></tr>');
            });
    }

    function escapeHtml(s) {
        return (s || '').toString().replace(/[&<>"']/g, function(m) {
            return ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[m]);
        });
    }

    // Statement modal handlers
    $(document).on('click', 'button[data-action="statement"]', function() {
        const id = $(this).data('id');
        const name = $(this).data('name');
        const currency = $(this).data('currency');
        $('#statementModalLabel').text('Account Statement - ' + name);
        $('#stmtAccountId').val(id);
        $('#stmtCurrency').val(currency || '');
        $('#statementModal').modal('show');
        fetchStatement();
    });

    $('#statementFilters').on('submit', function(e) {
        e.preventDefault();
        fetchStatement();
    });

    function fetchStatement() {
        const id = $('#stmtAccountId').val();
        const from = $('#stmtFrom').val();
        const to = $('#stmtTo').val();
        const currency = $('#stmtCurrency').val();
        const params = new URLSearchParams({ account_id: id });
        if (from) params.append('from', from);
        if (to) params.append('to', to);
        if (currency) params.append('currency', currency);
        const $stmtBody = $('#statementTable tbody');
        $stmtBody.html('<tr><td colspan="6" class="text-center text-muted">Loading...</td></tr>');
        fetch('api/accounts_statement.php?' + params.toString(), { credentials: 'same-origin' })
            .then(r => r.json())
            .then(res => {
                if (!res.success) throw new Error(res.error || 'Failed to load statement');
                $('#openingBalance').text('Opening: ' + Number(res.opening_balance || 0).toFixed(2));
                $('#closingBalance').text('Closing: ' + Number(res.closing_balance || 0).toFixed(2));
                const rows = (res.rows || []).map(row => {
                    const amt = Number(row.amount || 0).toFixed(2);
                    const run = Number(row.running_balance || 0).toFixed(2);
                    const typeBadge = row.type === 'credit' ? 'bg-success' : 'bg-danger';
                    const typeText = row.type === 'credit' ? 'Credit' : 'Debit';
                    return `
                        <tr>
                            <td>${escapeHtml(row.entry_date)}</td>
                            <td><span class="badge ${typeBadge}">${typeText}</span></td>
                            <td class="text-end">${amt}</td>
                            <td>${escapeHtml(row.currency)}</td>
                            <td>${escapeHtml(row.description || '')}</td>
                            <td class="text-end">${run}</td>
                        </tr>
                    `;
                }).join('');
                $stmtBody.html(rows || '<tr><td colspan="6" class="text-center text-muted">No entries in range.</td></tr>');
            })
            .catch(err => {
                $stmtBody.html('<tr><td colspan="6" class="text-center text-danger">' + (err.message || 'Failed to load statement') + '</td></tr>');
            });
    }

    $('#refreshAccounts').on('click', loadAccounts);
    loadAccounts();
})();
</script>


