<?php
// Security check
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: login.php");
    exit();
}

// CSRF Protection
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

require_once('includes/cache_handler.php');

try {
    $debug = filter_input(INPUT_GET, 'debug', FILTER_VALIDATE_INT) === 1;
    
    $cache = new DashboardCache();
    // Only use cache in non-debug mode
    $dashboardData = $debug ? null : $cache->getCachedData();
    
    if ($dashboardData === null) {
        $dashboardData = $cache->updateCache($db1, $debug);
    }
    
    if (!$dashboardData) {
        throw new Exception('Failed to retrieve dashboard data');
    }
    
    // Add debug information if requested
    if ($debug) {
        echo '<div class="alert alert-info">';
        echo '<h4>Cache Information</h4>';
        echo '<pre>';
        echo 'Cache Last Updated: ' . date('Y-m-d H:i:s', $dashboardData['timestamp']) . "\n";
        echo 'Cache Age: ' . (time() - $dashboardData['timestamp']) . " seconds\n";
        echo '</pre>';
        echo '</div>';
    }
    
    $topClients = $dashboardData['data'] ?? [];
    $summary = $dashboardData['summary'] ?? [];
    $insights = $dashboardData['insights'] ?? [];
    ?>
    <!-- Loading overlay -->
    <div id="loading-overlay" style="display: none;"> 
        <div class="spinner-border text-primary" role="status">
            <span class="visually-hidden">Loading...</span>
        </div>
    </div>

    <div class="dashboard-container">
        <!-- Stats Overview -->
        <div class="row g-4 mb-5">
            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-primary-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-people-fill fs-4 text-primary"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Total Active Clients</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['total_clients'] ?? 107); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-primary" style="width: 70%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-success-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-file-text-fill fs-4 text-success"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Total Nusuk Passports Processed</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['total_passports'] ?? 369947); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-success" style="width: 85%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-info-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-card-text fs-4 text-info"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Total Enjaz Processed</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['total_enjaz'] ?? 0); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-info" style="width: 80%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-primary-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-stack fs-4 text-primary"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Combined Total Documents</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['combined_total'] ?? 0); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-primary" style="width: 85%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-danger-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-exclamation-triangle-fill fs-4 text-danger"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Low Credit Clients</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['low_credit'] ?? 36); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-danger" style="width: 30%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-warning-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-clock-history fs-4 text-warning"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Inactive Clients</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['inactive_clients'] ?? 0); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-warning" style="width: 25%"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Insights Overview -->
        <div class="row g-4 mb-4">
            <?php 
                $topCountries = $insights['top_countries'] ?? [];
                $topCountryName = $topCountries[0]['country'] ?? 'N/A';
                $topCountryTotal = $topCountries[0]['total_docs'] ?? 0;
                $renewalBuckets = $insights['renewal_buckets'] ?? ['0-15'=>0,'16-45'=>0,'46-90'=>0];
                $usageDist = $insights['usage_distribution'] ?? ['0-50'=>0,'50-80'=>0,'80-90'=>0,'90-100'=>0];
                $totalClients = $summary['total_clients'] ?? 0;
            ?>
            <div class="col-xl-4 col-md-6">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-2">
                            <div class="stat-icon bg-info-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-geo-alt-fill fs-4 text-info"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Top Countries</h6>
                                <h5 class="card-title mb-0">Overview</h5>
                            </div>
                        </div>
                        <?php 
                            $topFive = array_slice($topCountries, 0, 5);
                            $maxCountryDocs = 0; 
                            foreach ($topFive as $c) { $maxCountryDocs = max($maxCountryDocs, (int)$c['total_docs']); }
                        ?>
                        <?php if (!empty($topFive)) { ?>
                            <div>
                                <?php foreach ($topFive as $c): ?>
                                    <div class="d-flex justify-content-between small">
                                        <span class="text-truncate" style="max-width: 60%;" title="<?php echo htmlspecialchars($c['country']); ?>">
                                            <?php echo htmlspecialchars($c['country']); ?>
                                        </span>
                                        <span class="ms-2 fw-semibold"><?php echo number_format((int)$c['total_docs']); ?></span>
                                    </div>
                                    <div class="progress mb-2" style="height:6px;">
                                        <div class="progress-bar bg-primary" style="width: <?php echo $maxCountryDocs ? (int)($c['total_docs']/$maxCountryDocs*100) : 0; ?>%"></div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php } else { ?>
                            <div class="text-muted small">No country data available.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="col-xl-4 col-md-6">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-2">
                            <div class="stat-icon bg-warning-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-calendar2-event fs-4 text-warning"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Renewals (days)</h6>
                                <h5 class="card-title mb-0">0-15: <?php echo (int)$renewalBuckets['0-15']; ?> • 16-45: <?php echo (int)$renewalBuckets['16-45']; ?> • 46-90: <?php echo (int)$renewalBuckets['46-90']; ?></h5>
                            </div>
                        </div>
                        <?php $renewTotal = max(1, array_sum($renewalBuckets)); ?>
                        <div class="d-flex justify-content-between text-muted small mb-2">
                            <span><span class="legend-dot bg-danger"></span>0–15</span>
                            <span><span class="legend-dot bg-warning"></span>16–45</span>
                            <span><span class="legend-dot bg-info"></span>46–90</span>
                        </div>
                        <div class="progress progress-renewals" style="height:10px;">
                            <div class="progress-bar bg-danger" title="0–15 days" style="width: <?php echo (int)($renewalBuckets['0-15']/$renewTotal*100); ?>%"></div>
                            <div class="progress-bar bg-warning" title="16–45 days" style="width: <?php echo (int)($renewalBuckets['16-45']/$renewTotal*100); ?>%"></div>
                            <div class="progress-bar bg-info" title="46–90 days" style="width: <?php echo (int)($renewalBuckets['46-90']/$renewTotal*100); ?>%"></div>
                        </div>
                        <div class="d-flex justify-content-between text-muted small mt-2">
                            <span><?php echo (int)$renewalBuckets['0-15']; ?></span>
                            <span><?php echo (int)$renewalBuckets['16-45']; ?></span>
                            <span><?php echo (int)$renewalBuckets['46-90']; ?></span>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-4 col-md-12">
                <div class="card border-0 shadow-sm h-100">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-2">
                            <div class="stat-icon bg-primary-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-speedometer2 fs-4 text-primary"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Usage Distribution</h6>
                                <h5 class="card-title mb-0">90-100%: <?php echo (int)$usageDist['90-100']; ?> of <?php echo (int)$totalClients; ?></h5>
                            </div>
                        </div>
                        <div class="d-flex justify-content-between text-muted small mb-1">
                            <span>0-50</span><span><?php echo (int)$usageDist['0-50']; ?></span>
                        </div>
                        <div class="progress mb-2" style="height:4px;">
                            <?php $maxClients = max(1, $totalClients); ?>
                            <div class="progress-bar bg-secondary" style="width: <?php echo (int)($usageDist['0-50']/$maxClients*100); ?>%"></div>
                        </div>
                        <div class="d-flex justify-content-between text-muted small mb-1">
                            <span>50-80</span><span><?php echo (int)$usageDist['50-80']; ?></span>
                        </div>
                        <div class="progress mb-2" style="height:4px;">
                            <div class="progress-bar bg-info" style="width: <?php echo (int)($usageDist['50-80']/$maxClients*100); ?>%"></div>
                        </div>
                        <div class="d-flex justify-content-between text-muted small mb-1">
                            <span>80-90</span><span><?php echo (int)$usageDist['80-90']; ?></span>
                        </div>
                        <div class="progress mb-2" style="height:4px;">
                            <div class="progress-bar bg-warning" style="width: <?php echo (int)($usageDist['80-90']/$maxClients*100); ?>%"></div>
                        </div>
                        <div class="d-flex justify-content-between text-muted small mb-1">
                            <span>90-100</span><span><?php echo (int)$usageDist['90-100']; ?></span>
                        </div>
                        <div class="progress" style="height:4px;">
                            <div class="progress-bar bg-danger" style="width: <?php echo (int)($usageDist['90-100']/$maxClients*100); ?>%"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Global Controls -->
        <div class="d-flex justify-content-between align-items-center mb-3 flex-wrap gap-2">
            <div class="d-flex align-items-center gap-2">
                <select id="managerFilter" class="form-select form-select-sm">
                    <option value="all">All Clients</option>
                    <option value="assigned">Manager Clients</option>
                    <option value="unassigned">Unassigned</option>
                    <?php
                    $managerNames = [];
                    foreach (($topClients ?? []) as $row) {
                        if (!empty($row['account_manager_id'])) {
                            $managerNames[(int)$row['account_manager_id']] = $row['AccountManagerName'] ?? ('Manager #' . (int)$row['account_manager_id']);
                        }
                    }
                    asort($managerNames);
                    foreach ($managerNames as $mid => $mname) {
                        echo '<option value="mgr_' . (int)$mid . '">' . htmlspecialchars($mname) . '</option>';
                    }
                    ?>
                </select>
                <button class="btn btn-sm btn-outline-primary" id="refreshData">
                    <i class="bi bi-arrow-clockwise"></i> Refresh
                </button>
            </div>
            <div class="text-muted small">
                <i class="bi bi-info-circle"></i> Use the "Show X entries" dropdown in each table to control how many clients are displayed
            </div>
        </div>

        <!-- Tab Navigation -->
        <ul class="nav nav-tabs mb-4" id="dashboardTabs" role="tablist">
            <li class="nav-item">
                <a class="nav-link active" id="overview-tab" data-bs-toggle="tab" href="#overview" role="tab">
                    Overview
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="countries-tab" data-bs-toggle="tab" href="#countries" role="tab">
                    Countries
                    <?php if (!empty($insights['top_countries'])): ?>
                        <span class="badge bg-secondary ms-1"><?php echo count($insights['top_countries']); ?></span>
                    <?php endif; ?>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="nationalities-tab" data-bs-toggle="tab" href="#nationalities" role="tab">
                    Nationalities
                    <?php if (!empty($insights['top_nationalities'])): ?>
                        <span class="badge bg-secondary ms-1"><?php echo count($insights['top_nationalities']); ?></span>
                    <?php endif; ?>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="low-credit-tab" data-bs-toggle="tab" href="#low-credit" role="tab">
                    Low Credit (<100)
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="critical-tab" data-bs-toggle="tab" href="#critical" role="tab">
                    Critical (<20%)
                    <?php if (count($dashboardData['critical_clients'] ?? []) > 0): ?>
                        <span class="badge bg-danger ms-1"><?php echo count($dashboardData['critical_clients']); ?></span>
                    <?php endif; ?>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="renewal-tab" data-bs-toggle="tab" href="#renewal" role="tab">
                    Renewal
                    <?php if (count($dashboardData['renewal_clients'] ?? []) > 0): ?>
                        <span class="badge bg-warning ms-1"><?php echo count($dashboardData['renewal_clients']); ?></span>
                    <?php endif; ?>
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="inactive-tab" data-bs-toggle="tab" href="#inactive" role="tab">
                    Inactive Clients
                    <?php if (count($dashboardData['inactive_clients'] ?? []) > 0): ?>
                        <span class="badge bg-warning ms-1"><?php echo count($dashboardData['inactive_clients']); ?></span>
                    <?php endif; ?>
                </a>
            </li>
        </ul>

        <!-- Tab Content -->
        <div class="tab-content">
            <!-- Overview Tab -->
            <div class="tab-pane fade show active" id="overview" role="tabpanel">
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center flex-wrap gap-2">
                        <h5 class="mb-0">Top Performing Clients</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover" id="topClientsTable">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>Country</th>
                                        <th>Passports</th>
                                        <th>Enjaz</th>
                                        <th>Total Docs</th>
                                        <th>Package</th>
                                        <th>Credit</th>
                                        <th>Account Manager</th>
                                        <th>Expiry Date</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($topClients as $row) { ?>
                                        <tr data-manager-id="<?php echo (int)($row['account_manager_id'] ?? 0); ?>">
                                            <td><?php echo htmlspecialchars($row['CompanyName']); ?></td>
                                            <td><?php echo htmlspecialchars($row['CountryName']); ?></td>
                                            <td><?php echo number_format($row['total_passports']); ?></td>
                                            <td><?php echo number_format($row['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($row['total_passports'] + $row['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($row['package'] ?? 0); ?></td>
                                            <td><?php echo number_format($row['credit'] ?? 0); ?></td>
                                            <td><?php echo htmlspecialchars($row['AccountManagerName'] ?? '—'); ?></td>
                                            <td style="font-size: 0.875rem;"><?php echo isset($row['Renew_Date']) ? date('Y-m-d', strtotime($row['Renew_Date'])) : 'N/A'; ?></td>
                                            <td>
                                                <a href="?page=edit_client&id=<?php echo $row['Id']; ?>" 
                                                   class="btn btn-xs btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </a>
                                                <a href="download_passports.php?client_id=<?php echo $row['Id']; ?>" 
                                                   class="btn btn-xs btn-success">
                                                    <i class="bi bi-download"></i> Passports
                                                </a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                        <div class="row mt-4 g-4">
                            <div class="col-lg-4">
                                <div class="card border-0 shadow-sm h-100">
                                    <div class="card-header d-flex justify-content-between align-items-center">
                                        <h6 class="mb-0">Top Countries</h6>
                                    </div>
                                    <div class="card-body">
                                        <?php 
                                        $maxCountry = 0; 
                                        foreach (($insights['top_countries'] ?? []) as $c) { $maxCountry = max($maxCountry, (int)$c['total_docs']); }
                                        ?>
                                        <div class="table-responsive">
                                            <table class="table align-middle">
                                                <thead>
                                                    <tr>
                                                        <th>Country</th>
                                                        <th class="text-end">Total Docs</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                <?php foreach (($insights['top_countries'] ?? []) as $country): ?>
                                                    <tr>
                                                        <td>
                                                            <div class="d-flex flex-column">
                                                                <strong><?php echo htmlspecialchars($country['country']); ?></strong>
                                                                <div class="progress mt-1" style="height:4px;">
                                                                    <div class="progress-bar bg-primary" style="width: <?php echo $maxCountry ? (int)($country['total_docs']/$maxCountry*100) : 0; ?>%"></div>
                                                                </div>
                                                            </div>
                                                        </td>
                                                        <td class="text-end fw-semibold"><?php echo number_format((int)$country['total_docs']); ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                                <?php if (empty($insights['top_countries'])): ?>
                                                    <tr><td colspan="2" class="text-muted">No country data available.</td></tr>
                                                <?php endif; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-lg-4">
                                <div class="card border-0 shadow-sm h-100">
                                    <div class="card-header d-flex justify-content-between align-items-center">
                                        <h6 class="mb-0">Top Nationalities</h6>
                                    </div>
                                    <div class="card-body">
                                        <?php 
                                        $maxNationality = 0; 
                                        foreach (($insights['top_nationalities'] ?? []) as $n) { $maxNationality = max($maxNationality, (int)$n['total_passports']); }
                                        ?>
                                        <div class="table-responsive">
                                            <table class="table align-middle">
                                                <thead>
                                                    <tr>
                                                        <th>Nationality</th>
                                                        <th class="text-end">Passports</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                <?php foreach (($insights['top_nationalities'] ?? []) as $nationality): ?>
                                                    <tr>
                                                        <td>
                                                            <div class="d-flex flex-column">
                                                                <strong><?php echo htmlspecialchars($nationality['nationality']); ?></strong>
                                                                <div class="progress mt-1" style="height:4px;">
                                                                    <div class="progress-bar bg-success" style="width: <?php echo $maxNationality ? (int)($nationality['total_passports']/$maxNationality*100) : 0; ?>%"></div>
                                                                </div>
                                                            </div>
                                                        </td>
                                                        <td class="text-end fw-semibold"><?php echo number_format((int)$nationality['total_passports']); ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                                <?php if (empty($insights['top_nationalities'])): ?>
                                                    <tr><td colspan="2" class="text-muted">No nationality data available.</td></tr>
                                                <?php endif; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-lg-4">
                                <div class="card border-0 shadow-sm h-100">
                                    <div class="card-header d-flex justify-content-between align-items-center">
                                        <h6 class="mb-0">Recent Activity</h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="table-responsive">
                                            <table class="table align-middle">
                                                <thead>
                                                    <tr>
                                                        <th>Client</th>
                                                        <th>Country</th>
                                                        <th>Last Access</th>
                                                        <th class="text-end">Days</th>
                                                    </tr>
                                                </thead>
                                                <tbody>
                                                <?php foreach (($insights['recent_activity'] ?? []) as $ra): ?>
                                                    <tr>
                                                        <td><?php echo htmlspecialchars($ra['CompanyName'] ?? ''); ?></td>
                                                        <td><?php echo htmlspecialchars($ra['CountryName'] ?? ''); ?></td>
                                                        <td style="font-size: 0.875rem;">
                                                            <?php echo isset($ra['last_access_date']) && $ra['last_access_date'] ? date('Y-m-d H:i', strtotime($ra['last_access_date'])) : 'N/A'; ?>
                                                        </td>
                                                        <td class="text-end"><?php echo isset($ra['days_since_last_access']) ? (int)$ra['days_since_last_access'] : '-'; ?></td>
                                                    </tr>
                                                <?php endforeach; ?>
                                                <?php if (empty($insights['recent_activity'])): ?>
                                                    <tr><td colspan="4" class="text-muted">No recent activity available.</td></tr>
                                                <?php endif; ?>
                                                </tbody>
                                            </table>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="text-muted small mt-3">
                            Cache updated: <?php echo isset($dashboardData['timestamp']) ? date('Y-m-d H:i:s', $dashboardData['timestamp']) : 'N/A'; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Countries Tab -->
            <div class="tab-pane fade" id="countries" role="tabpanel" aria-labelledby="countries-tab">
                <div class="card shadow-sm mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Top Countries by Total Documents</h5>
                        <span class="text-muted small">
                            Total Docs: <?php echo number_format((int)($summary['combined_total'] ?? 0)); ?>
                        </span>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover align-middle">
                                <thead>
                                    <tr>
                                        <th style="width: 80px;">#</th>
                                        <th>Country</th>
                                        <th class="text-end">Total Docs</th>
                                        <th class="text-end">Share</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $countriesList = $insights['top_countries'] ?? [];
                                    $grandTotal = max(1, (int)($summary['combined_total'] ?? 0));
                                    $rank = 1;
                                    foreach ($countriesList as $c): 
                                        $docs = (int)($c['total_docs'] ?? 0);
                                        $pct = $docs > 0 ? ($docs / $grandTotal) * 100 : 0;
                                    ?>
                                        <tr>
                                            <td><?php echo $rank++; ?></td>
                                            <td><?php echo htmlspecialchars($c['country'] ?? ''); ?></td>
                                            <td class="text-end fw-semibold"><?php echo number_format($docs); ?></td>
                                            <td class="text-end">
                                                <div class="d-inline-flex align-items-center" style="min-width:120px;">
                                                    <div class="progress flex-grow-1 me-2" style="height:6px;">
                                                        <div class="progress-bar bg-primary" style="width: <?php echo (int)$pct; ?>%"></div>
                                                    </div>
                                                    <span class="text-muted small"><?php echo number_format($pct, 1); ?>%</span>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                    <?php if (empty($countriesList)): ?>
                                        <tr>
                                            <td colspan="4" class="text-center text-muted">No country data available.</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Nationalities Tab -->
            <div class="tab-pane fade" id="nationalities" role="tabpanel" aria-labelledby="nationalities-tab">
                <div class="card shadow-sm mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Top Nationalities by Passport Count</h5>
                        <span class="text-muted small">
                            Total Passports: <?php echo number_format(array_sum(array_column($insights['top_nationalities'] ?? [], 'total_passports'))); ?>
                        </span>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover align-middle">
                                <thead>
                                    <tr>
                                        <th style="width: 80px;">#</th>
                                        <th>Nationality</th>
                                        <th class="text-end">Total Passports</th>
                                        <th class="text-end">Share</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $nationalitiesList = $insights['top_nationalities'] ?? [];
                                    $grandTotalPassports = max(1, array_sum(array_column($nationalitiesList, 'total_passports')));
                                    $rank = 1;
                                    foreach ($nationalitiesList as $n): 
                                        $passports = (int)($n['total_passports'] ?? 0);
                                        $pct = $passports > 0 ? ($passports / $grandTotalPassports) * 100 : 0;
                                    ?>
                                        <tr>
                                            <td><?php echo $rank++; ?></td>
                                            <td><?php echo htmlspecialchars($n['nationality'] ?? ''); ?></td>
                                            <td class="text-end fw-semibold"><?php echo number_format($passports); ?></td>
                                            <td class="text-end">
                                                <div class="d-inline-flex align-items-center" style="min-width:120px;">
                                                    <div class="progress flex-grow-1 me-2" style="height:6px;">
                                                        <div class="progress-bar bg-success" style="width: <?php echo (int)$pct; ?>%"></div>
                                                    </div>
                                                    <span class="text-muted small"><?php echo number_format($pct, 1); ?>%</span>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                    <?php if (empty($nationalitiesList)): ?>
                                        <tr>
                                            <td colspan="4" class="text-center text-muted">No nationality data available.</td>
                                        </tr>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Low Credit Clients Tab -->
            <div class="tab-pane fade" id="low-credit" role="tabpanel" aria-labelledby="low-credit-tab">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover" id="lowCreditTable">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>Country</th>
                                        <th>Passports</th>
                                        <th>Enjaz</th>
                                        <th>Total Docs</th>
                                        <th>Package</th>
                                        <th>Remaining Credit</th>
                                        <th>Account Manager</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    // Get low credit clients from the cached data
                                    $lowCreditClients = $dashboardData['low_credit_clients'] ?? [];
                                    
                                    foreach ($lowCreditClients as $client) {
                                        // Calculate percentage of credit remaining
                                        $creditPercentage = ($client['package'] > 0) ? 
                                            ($client['credit'] / $client['package']) * 100 : 0;
                                        
                                        // Determine alert level based on remaining credit
                                        $rowClass = '';
                                        if ($client['credit'] <= 0) {
                                            $rowClass = 'table-danger';
                                        } elseif ($client['credit'] <= 50) {
                                            $rowClass = 'table-warning';
                                        }
                                    ?>
                                        <tr class="<?php echo $rowClass; ?>" data-manager-id="<?php echo (int)($client['account_manager_id'] ?? 0); ?>">
                                            <td><?php echo htmlspecialchars($client['CompanyName']); ?></td>
                                            <td><?php echo htmlspecialchars($client['CountryName']); ?></td>
                                            <td><?php echo number_format($client['total_passports']); ?></td>
                                            <td><?php echo number_format($client['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($client['total_passports'] + $client['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($client['package']); ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <span class="me-2"><?php echo number_format($client['credit']); ?></span>
                                                    <div class="progress flex-grow-1" style="height: 6px;">
                                                        <div class="progress-bar <?php echo $client['credit'] <= 50 ? 'bg-danger' : 'bg-warning'; ?>" 
                                                             role="progressbar" 
                                                             style="width: <?php echo min(100, max(0, $creditPercentage)); ?>%" 
                                                             aria-valuenow="<?php echo $client['credit']; ?>" 
                                                             aria-valuemin="0" 
                                                             aria-valuemax="<?php echo $client['package']; ?>">
                                                        </div>
                                                    </div>
                                                </div>
                                            </td>
                                            <td><?php echo htmlspecialchars($client['AccountManagerName'] ?? '—'); ?></td>
                                            <td>
                                                <a href="?page=edit_client&id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </a>
                                                <a href="download_passports.php?client_id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-success">
                                                    <i class="bi bi-download"></i> Download
                                                </a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                    <?php if (empty($lowCreditClients)) { ?>
                                        <tr>
                                            <td colspan="9" class="text-center">No clients with low credit found.</td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Critical Clients Tab -->
            <div class="tab-pane fade" id="critical" role="tabpanel" aria-labelledby="critical-tab">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover" id="criticalTable">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>Country</th>
                                        <th>Passports</th>
                                        <th>Enjaz</th>
                                        <th>Total Docs</th>
                                        <th>Package</th>
                                        <th>Remaining Credit</th>
                                        <th>Usage %</th>
                                        <th>Account Manager</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $criticalClients = $dashboardData['critical_clients'] ?? [];
                                    
                                    foreach ($criticalClients as $client) {
                                        $usagePercentage = ($client['package'] > 0) ? 
                                            (($client['package'] - $client['credit']) / $client['package']) * 100 : 100;
                                        
                                        $rowClass = $usagePercentage >= 90 ? 'table-danger' : 'table-warning';
                                    ?>
                                        <tr class="<?php echo $rowClass; ?>" data-manager-id="<?php echo (int)($client['account_manager_id'] ?? 0); ?>">
                                            <td><?php echo htmlspecialchars($client['CompanyName']); ?></td>
                                            <td><?php echo htmlspecialchars($client['CountryName']); ?></td>
                                            <td><?php echo number_format($client['total_passports']); ?></td>
                                            <td><?php echo number_format($client['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($client['total_passports'] + $client['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($client['package']); ?></td>
                                            <td><?php echo number_format($client['credit']); ?></td>
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="progress flex-grow-1" style="height: 6px;">
                                                        <div class="progress-bar bg-danger" 
                                                             role="progressbar" 
                                                             style="width: <?php echo min(100, $usagePercentage); ?>%" 
                                                             aria-valuenow="<?php echo $usagePercentage; ?>" 
                                                             aria-valuemin="0" 
                                                             aria-valuemax="100">
                                                        </div>
                                                    </div>
                                                    <span class="ms-2"><?php echo number_format($usagePercentage, 1); ?>%</span>
                                                </div>
                                            </td>
                                            <td><?php echo htmlspecialchars($client['AccountManagerName'] ?? '—'); ?></td>
                                            <td>
                                                <a href="?page=edit_client&id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </a>
                                                <a href="download_passports.php?client_id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-success">
                                                    <i class="bi bi-download"></i> Download
                                                </a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                    <?php if (empty($criticalClients)) { ?>
                                        <tr>
                                            <td colspan="10" class="text-center">No clients with critical credit levels found.</td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Renewal Clients Tab -->
            <div class="tab-pane fade" id="renewal" role="tabpanel" aria-labelledby="renewal-tab">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover" id="renewalTable">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>Country</th>
                                        <th>Expiry Date</th>
                                        <th>Days Until Expiry</th>
                                        <th>Status</th>
                                        <th>Account Manager</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $renewalClients = $dashboardData['renewal_clients'] ?? [];
                                    
                                    foreach ($renewalClients as $client) {
                                        $daysUntilExpiry = $client['days_until_expiry'];
                                        
                                        // Determine row styling based on expiry proximity
                                        $rowClass = '';
                                        $statusText = '';
                                        $statusClass = '';
                                        
                                        if ($daysUntilExpiry <= 15) {
                                            $rowClass = 'table-danger';
                                            $statusText = 'Critical';
                                            $statusClass = 'badge bg-danger';
                                        } elseif ($daysUntilExpiry <= 45) {
                                            $rowClass = 'table-warning';
                                            $statusText = 'Warning';
                                            $statusClass = 'badge bg-warning';
                                        }
                                    ?>
                                        <tr class="<?php echo $rowClass; ?>" data-manager-id="<?php echo (int)($client['account_manager_id'] ?? 0); ?>">
                                            <td><?php echo htmlspecialchars($client['CompanyName']); ?></td>
                                            <td><?php echo htmlspecialchars($client['CountryName']); ?></td>
                                            <td style="font-size: 0.875rem;"><?php echo date('Y-m-d', strtotime($client['Renew_Date'])); ?></td>
                                            <td>
                                                <strong><?php echo $daysUntilExpiry; ?></strong> days
                                            </td>
                                            <td>
                                                <span class="<?php echo $statusClass; ?>"><?php echo $statusText; ?></span>
                                            </td>
                                            <td><?php echo htmlspecialchars($client['AccountManagerName'] ?? '—'); ?></td>
                                            <td>
                                                <a href="?page=edit_client&id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                    <?php if (empty($renewalClients)) { ?>
                                        <tr>
                                            <td colspan="7" class="text-center">No clients with upcoming renewals found.</td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Inactive Clients Tab -->
            <div class="tab-pane fade" id="inactive" role="tabpanel" aria-labelledby="inactive-tab">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover" id="inactiveTable">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>Country</th>
                                        <th>Passports</th>
                                        <th>Enjaz</th>
                                        <th>Total Docs</th>
                                        <th>Package</th>
                                        <th>Remaining Credit</th>
                                        <th>Last Access Date</th>
                                        <th>Days Since Last Access</th>
                                        <th>Account Manager</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $inactiveClients = $dashboardData['inactive_clients'] ?? [];
                                    
                                    foreach ($inactiveClients as $client) {
                                        $daysSinceLastAccess = $client['days_since_last_access'] ?? 0;
                                        
                                        // Determine row styling based on inactivity period
                                        $rowClass = '';
                                        $statusText = '';
                                        $statusClass = '';
                                        
                                        if ($daysSinceLastAccess >= 30) {
                                            $rowClass = 'table-danger';
                                            $statusText = 'Critical (30+ days)';
                                            $statusClass = 'badge bg-danger';
                                        } elseif ($daysSinceLastAccess >= 15) {
                                            $rowClass = 'table-warning';
                                            $statusText = 'Warning (15+ days)';
                                            $statusClass = 'badge bg-warning';
                                        }
                                    ?>
                                        <tr class="<?php echo $rowClass; ?>" data-manager-id="<?php echo (int)($client['account_manager_id'] ?? 0); ?>">
                                            <td><?php echo htmlspecialchars($client['CompanyName']); ?></td>
                                            <td><?php echo htmlspecialchars($client['CountryName']); ?></td>
                                            <td><?php echo number_format($client['total_passports']); ?></td>
                                            <td><?php echo number_format($client['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($client['total_passports'] + $client['enjaz_passports']); ?></td>
                                            <td><?php echo number_format($client['package'] ?? 0); ?></td>
                                            <td><?php echo number_format($client['credit'] ?? 0); ?></td>
                                            <td style="font-size: 0.875rem;">
                                                <?php echo isset($client['last_access_date']) ? date('Y-m-d H:i:s', strtotime($client['last_access_date'])) : 'N/A'; ?>
                                            </td>
                                            <td>
                                                <strong><?php echo $daysSinceLastAccess; ?></strong> days
                                            </td>
                                            <td><?php echo htmlspecialchars($client['AccountManagerName'] ?? '—'); ?></td>
                                            <td>
                                                <span class="<?php echo $statusClass; ?>"><?php echo $statusText; ?></span>
                                            </td>
                                            <td>
                                                <a href="?page=edit_client&id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </a>
                                                <a href="download_passports.php?client_id=<?php echo $client['Id']; ?>" 
                                                   class="btn btn-xs btn-success">
                                                    <i class="bi bi-download"></i> Download
                                                </a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                    <?php if (empty($inactiveClients)) { ?>
                                        <tr>
                                            <td colspan="12" class="text-center text-muted">No inactive clients found.</td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Enhanced DataTables Initialization -->
        <script>
        $(document).ready(function() {
            console.log('Document ready executed'); // Debug log
            
            // Global manager filter state
            let managerFilterValue = 'all';

            // DataTables custom search function for manager filtering
            let currentManagerFilter = 'all';
            
            $.fn.dataTable.ext.search.push(function(settings, data, dataIndex) {
                if (!currentManagerFilter || currentManagerFilter === 'all') return true;

                const api = new $.fn.dataTable.Api(settings);
                const rowNode = api.row(dataIndex).node();
                const mid = parseInt($(rowNode).attr('data-manager-id') || '0', 10);

                if (currentManagerFilter === 'assigned') return mid > 0;
                if (currentManagerFilter === 'unassigned') return mid === 0;
                if (currentManagerFilter.startsWith('mgr_')) {
                    const target = parseInt(currentManagerFilter.replace('mgr_', ''), 10);
                    return mid === target;
                }
                return true;
            });

            // Initialize DataTables with improved configuration
            const topClientsTable = $('#topClientsTable').DataTable({
                order: [[4, 'desc']], // Sort by Total Docs descending
                pageLength: 100,
                responsive: true,
                language: {
                    search: "Search clients:",
                    lengthMenu: "Show _MENU_ clients per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ clients"
                },
                dom: 'lfrtip',
                lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                stateSave: false
            });

            const lowCreditTable = $('#lowCreditTable').DataTable({
                order: [[4, 'asc']],
                pageLength: 100,
                responsive: true,
                language: {
                    search: "Search clients:",
                    lengthMenu: "Show _MENU_ clients per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ clients"
                },
                dom: 'lfrtip',
                lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                stateSave: false
            });

            // Initialize DataTable for inactive clients
            const inactiveTable = $('#inactiveTable').DataTable({
                order: [[8, 'desc']], // Sort by days since last access (descending)
                pageLength: 100,
                responsive: true,
                language: {
                    search: "Search inactive clients:",
                    lengthMenu: "Show _MENU_ clients per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ clients"
                },
                dom: 'lfrtip',
                lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
                stateSave: false
            });

            // Apply current filter on initial draw (no-op for 'all')
            topClientsTable.draw();
            lowCreditTable.draw();
            inactiveTable.draw();
            
            // Debug: Check if length selectors are present
            setTimeout(function() {
                console.log('Checking for length selectors...');
                
                // Check all DataTables length selectors
                const allLengthSelectors = $('.dataTables_length select');
                console.log('Total length selectors found:', allLengthSelectors.length);
                
                allLengthSelectors.each(function(index) {
                    console.log('Length selector ' + (index + 1) + ':', $(this));
                });
                
                // Force show all length selectors
                $('.dataTables_length').show();
                $('.dataTables_length select').show();
            }, 1000);

            // Add debug logging to refresh button
            $('#refreshData').on('click', function(e) {
                console.log('Refresh button clicked'); // Debug log
                e.preventDefault();
                
                const $btn = $(this);
                const $icon = $btn.find('i');
                
                console.log('Starting AJAX request'); // Debug log
                
                $btn.prop('disabled', true);
                $icon.addClass('rotate');
                $('#loading-overlay').show();

                // Log CSRF token (but not in production!)
                console.log('CSRF Token:', '<?php echo $_SESSION['csrf_token']; ?>');

                $.ajax({
                    url: 'cron/update_dashboard_cache.php',
                    method: 'POST',
                    data: {
                        csrf_token: '<?php echo $_SESSION['csrf_token']; ?>'
                    },
                    dataType: 'json',
                    beforeSend: function() {
                        console.log('AJAX beforeSend triggered'); // Debug log
                    }
                })
                .done(function(response) {
                    console.log('AJAX success:', response); // Debug log
                    if (response.success) {
                        window.location.reload();
                    } else {
                        showAlert(response.message || 'Failed to refresh data', 'danger');
                    }
                })
                .fail(function(jqXHR, textStatus, errorThrown) {
                    console.log('AJAX failed:', textStatus, errorThrown); // Debug log
                    showAlert('Failed to refresh data. Please try again.', 'danger');
                    console.error('Refresh failed:', jqXHR.responseText);
                })
                .always(function() {
                    console.log('AJAX request completed'); // Debug log
                    $btn.prop('disabled', false);
                    $icon.removeClass('rotate');
                    $('#loading-overlay').hide();
                });
            });

            // Manager filter applies to all tabs (non-DataTables tables)
            function applyManagerFilter($rows, predicate) {
                $rows.each(function() {
                    const mid = parseInt($(this).attr('data-manager-id') || '0', 10);
                    $(this).toggle(predicate(mid));
                });
            }

            $('#managerFilter').on('change', function() {
                const val = $(this).val();
                console.log('managerFilter change:', val);
                managerFilterValue = val;
                currentManagerFilter = val;
                
                // Apply filter to non-DataTables tables (manual show/hide)
                const nonDataTables = [
                    $('#criticalTable tbody tr'),
                    $('#renewalTable tbody tr')
                ];
                
                if (val === 'all') { 
                    nonDataTables.forEach($r => $r.show()); 
                }
                else if (val === 'assigned') { 
                    nonDataTables.forEach($r => applyManagerFilter($r, mid => mid > 0)); 
                }
                else if (val === 'unassigned') { 
                    nonDataTables.forEach($r => applyManagerFilter($r, mid => mid === 0)); 
                }
                else if (val && val.startsWith('mgr_')) {
                    const target = parseInt(val.replace('mgr_', ''), 10);
                    console.log('Manager filter applied for manager ID:', target);
                    nonDataTables.forEach($r => applyManagerFilter($r, mid => mid === target));
                }
                
                // Apply filter to DataTables using custom search function
                if ($.fn.dataTable.isDataTable('#topClientsTable')) {
                    console.log('Redrawing topClientsTable with manager filter:', val);
                    console.log('TopClientsTable page length before draw:', topClientsTable.page.len());
                    topClientsTable.draw();
                    // Ensure page length is maintained
                    if (topClientsTable.page.len() !== 100) {
                        console.log('Resetting topClientsTable page length to 100');
                        topClientsTable.page.len(100).draw();
                    }
                    console.log('TopClientsTable page length after draw:', topClientsTable.page.len());
                    console.log('TopClientsTable visible rows:', topClientsTable.rows({search: 'applied'}).count());
                }
                if ($.fn.dataTable.isDataTable('#lowCreditTable')) {
                    console.log('Redrawing lowCreditTable with manager filter:', val);
                    console.log('LowCreditTable page length before draw:', lowCreditTable.page.len());
                    lowCreditTable.draw();
                    // Ensure page length is maintained
                    if (lowCreditTable.page.len() !== 100) {
                        console.log('Resetting lowCreditTable page length to 100');
                        lowCreditTable.page.len(100).draw();
                    }
                    console.log('LowCreditTable page length after draw:', lowCreditTable.page.len());
                    console.log('LowCreditTable visible rows:', lowCreditTable.rows({search: 'applied'}).count());
                }
                if ($.fn.dataTable.isDataTable('#inactiveTable')) {
                    console.log('Redrawing inactiveTable with manager filter:', val);
                    console.log('InactiveTable page length before draw:', inactiveTable.page.len());
                    inactiveTable.draw();
                    // Ensure page length is maintained
                    if (inactiveTable.page.len() !== 100) {
                        console.log('Resetting inactiveTable page length to 100');
                        inactiveTable.page.len(100).draw();
                    }
                    console.log('InactiveTable page length after draw:', inactiveTable.page.len());
                    console.log('InactiveTable visible rows:', inactiveTable.rows({search: 'applied'}).count());
                }
            });

            // Apply default filter once on load
            $('#managerFilter').trigger('change');

            // Helper function to show alerts
            function showAlert(message, type) {
                const alert = `
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>`;
                $('.dashboard-container').prepend(alert);
            }
        });
        </script>

        <style>
        .rotate {
            animation: rotate 1s linear infinite;
        }
        @keyframes rotate {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        #loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 9999;
        }
        .dashboard-container {
            padding: 2rem;
            background-color: #f8f9fa;
        }

        .stat-card {
            transition: transform 0.2s ease-in-out;
            background: white;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-icon {
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .progress {
            background-color: #e9ecef;
            border-radius: 2px;
        }

        /* Custom color utilities */
        .bg-primary-subtle {
            background-color: rgba(13, 110, 253, 0.1);
        }

        .bg-success-subtle {
            background-color: rgba(25, 135, 84, 0.1);
        }

        .bg-danger-subtle {
            background-color: rgba(220, 53, 69, 0.1);
        }

        .bg-info-subtle {
            background-color: rgba(13, 202, 240, 0.1);
        }

        .bg-warning-subtle {
            background-color: rgba(255, 193, 7, 0.1);
        }

        /* Responsive font sizes */
        @media (max-width: 768px) {
            .card-title {
                font-size: 1.5rem;
            }
        }

        /* DataTables length selector styling */
        .dataTables_length {
            margin-bottom: 1rem;
            display: block !important;
            visibility: visible !important;
        }
        
        .dataTables_length select {
            margin: 0 0.5rem;
            padding: 0.25rem 0.5rem;
            border: 1px solid #ced4da;
            border-radius: 0.25rem;
            background-color: white;
            display: inline-block !important;
            visibility: visible !important;
        }
        
        .dataTables_length label {
            font-weight: 500;
            margin-bottom: 0;
            display: inline-block !important;
            visibility: visible !important;
        }
        
        /* Ensure DataTables controls are visible */
        .dataTables_wrapper .dataTables_length,
        .dataTables_wrapper .dataTables_filter,
        .dataTables_wrapper .dataTables_info,
        .dataTables_wrapper .dataTables_paginate {
            display: block !important;
            visibility: visible !important;
        }

        /* Smaller action buttons */
        .btn-xs {
            padding: 0.125rem 0.25rem;
            font-size: 0.75rem;
            line-height: 1.2;
            border-radius: 0.2rem;
        }
        
        .btn-xs i {
            font-size: 0.75rem;
        }
        </style>
    </div>

    <?php
    } catch(Exception $e) {
        // Store the error message
        $errorMessage = $e->getMessage();
        $errorType = get_class($e);
        
        // Check database connection
        $dbConnectionStatus = "Unknown";
        try {
            if (isset($db1) && $db1 instanceof PDO) {
                $stmt = $db1->query("SELECT 1");
                $dbConnectionStatus = $stmt ? "Connected" : "Error";
            } else {
                $dbConnectionStatus = "Not initialized";
            }
        } catch (Exception $dbError) {
            $dbConnectionStatus = "Error: " . $dbError->getMessage();
        }
        
        // Determine what to show based on debug mode
        $displayMessage = $debug ? 
            htmlspecialchars($errorMessage) : 
            'An error occurred while fetching the dashboard data.';
        
        echo '<div class="alert alert-danger">
                <h4>Dashboard Error</h4>
                <p>' . $displayMessage . '</p>';
                
        if ($debug) {
            echo '<div class="mt-3">
                    <h5>Debug Information</h5>
                    <dl class="row">
                        <dt class="col-sm-3">Error Type:</dt>
                        <dd class="col-sm-9">' . htmlspecialchars($errorType) . '</dd>
                        
                        <dt class="col-sm-3">Full Error:</dt>
                        <dd class="col-sm-9">' . htmlspecialchars($errorMessage) . '</dd>
                        
                        <dt class="col-sm-3">DB Status:</dt>
                        <dd class="col-sm-9">' . htmlspecialchars($dbConnectionStatus) . '</dd>
                        
                        <dt class="col-sm-3">Cache Path:</dt>
                        <dd class="col-sm-9">' . htmlspecialchars(dirname(__FILE__) . '/../cache/') . '</dd>
                    </dl>
                </div>';
        }
                
        echo '<button class="btn btn-primary mt-2" id="errorRefreshData">
                <i class="bi bi-arrow-clockwise"></i> Refresh Dashboard Data
            </button>
            </div>';
            
        echo '<div id="loading-overlay" style="display: none;">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading...</span>
                </div>
              </div>';
        error_log("Dashboard Error: " . $e->getMessage());
        
        // Add script for the error refresh button
        echo '<script>
            document.addEventListener("DOMContentLoaded", function() {
                $("#errorRefreshData").on("click", function(e) {
                    e.preventDefault();
                    
                    const $btn = $(this);
                    const $icon = $btn.find("i");
                    
                    $btn.prop("disabled", true);
                    $icon.addClass("rotate");
                    $("#loading-overlay").show();
                    
                    $.ajax({
                        url: "cron/update_dashboard_cache.php",
                        method: "POST",
                        data: {
                            csrf_token: "' . ($_SESSION['csrf_token'] ?? '') . '"
                        },
                        dataType: "json"
                    })
                    .done(function(response) {
                        if (response.success) {
                            window.location.reload();
                        } else {
                            alert(response.message || "Failed to refresh data");
                        }
                    })
                    .fail(function(jqXHR, textStatus, errorThrown) {
                        alert("Failed to refresh data. Please try again.");
                        console.error("Refresh failed:", jqXHR.responseText);
                    })
                    .always(function() {
                        $btn.prop("disabled", false);
                        $icon.removeClass("rotate");
                        $("#loading-overlay").hide();
                    });
                });
            });
        </script>';
    }
    ?>

    <!-- Move the script to a separate block that will be included after jQuery -->
    <?php ob_start(); ?>
    <script>
        $(document).ready(function() {
            console.log('Document ready executed');
            
            $('#refreshData').on('click', function(e) {
                console.log('Refresh button clicked');
                e.preventDefault();
                
                const $btn = $(this);
                const $icon = $btn.find('i');
                
                console.log('Starting AJAX request');
                
                $btn.prop('disabled', true);
                $icon.addClass('rotate');
                $('#loading-overlay').show();

                $.ajax({
                    url: 'cron/update_dashboard_cache.php',
                    method: 'POST',
                    data: {
                        csrf_token: '<?php echo $_SESSION['csrf_token']; ?>'
                    },
                    dataType: 'json',
                    beforeSend: function() {
                        console.log('AJAX beforeSend triggered');
                    }
                })
                .done(function(response) {
                    console.log('AJAX success:', response);
                    if (response.success) {
                        window.location.reload();
                    } else {
                        showAlert(response.message || 'Failed to refresh data', 'danger');
                    }
                })
                .fail(function(jqXHR, textStatus, errorThrown) {
                    console.log('AJAX failed:', textStatus, errorThrown);
                    showAlert('Failed to refresh data. Please try again.', 'danger');
                    console.error('Refresh failed:', jqXHR.responseText);
                })
                .always(function() {
                    console.log('AJAX request completed');
                    $btn.prop('disabled', false);
                    $icon.removeClass('rotate');
                    $('#loading-overlay').hide();
                });
            });

            function showAlert(message, type) {
                const alert = `
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>`;
                $('.dashboard-container').prepend(alert);
            }
        });
    </script>
    <?php 
    $dashboardScript = ob_get_clean();
    // Add this script to a variable that will be included after jQuery in index.php
    if (!isset($pageScripts)) $pageScripts = '';
    $pageScripts .= $dashboardScript;
    ?>
