<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get invoice ID from URL
$invoice_id = $_GET['id'] ?? 0;

if (!$invoice_id) {
    echo '<div class="alert alert-danger">Invoice ID is required</div>';
    return;
}

// Get invoice details
$invoice = null;
$invoice_items = [];
$clients = [];
$applications = [];

try {
    // Get invoice header
    $stmt = $db1->prepare("
        SELECT i.*, c.CompanyName as client_name
        FROM invoices i
        LEFT JOIN tbl_clients c ON i.client_id = c.Id
        WHERE i.id = ?
    ");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$invoice) {
        echo '<div class="alert alert-danger">Invoice not found</div>';
        return;
    }
    
    // Check if invoice is editable (only draft invoices can be edited)
    if ($invoice['status'] !== 'draft') {
        echo '<div class="alert alert-warning">Only draft invoices can be edited</div>';
        echo '<a href="?page=invoice_view&id=' . $invoice_id . '" class="btn btn-secondary">View Invoice</a>';
        return;
    }
    
    // Get invoice items
    $stmt = $db1->prepare("
        SELECT ii.*, a.Application_Name
        FROM invoice_items ii
        LEFT JOIN tbl_applications a ON ii.application_id = a.id
        WHERE ii.invoice_id = ?
        ORDER BY ii.id
    ");
    $stmt->execute([$invoice_id]);
    $invoice_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get clients for dropdown
    $stmt = $db1->prepare("SELECT Id, CompanyName FROM tbl_clients ORDER BY CompanyName");
    $stmt->execute();
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get applications for dropdown
    $stmt = $db1->prepare("SELECT id, Application_Name FROM tbl_applications WHERE visible = 1 ORDER BY Application_Name");
    $stmt->execute();
    $applications = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    echo '<div class="alert alert-danger">Error loading invoice: ' . htmlspecialchars($e->getMessage()) . '</div>';
    return;
}
?>

<!-- Invoice Edit Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Edit Invoice #<?= htmlspecialchars($invoice['invoice_number']) ?></h2>
    <div>
        <a href="?page=invoice_view&id=<?= $invoice['id'] ?>" class="btn btn-secondary">View Invoice</a>
        <a href="?page=invoices" class="btn btn-outline-secondary">Back to Invoices</a>
    </div>
</div>

<!-- Edit Form -->
<form id="invoiceEditForm" method="POST" action="api/invoices_update.php">
    <input type="hidden" name="invoice_id" value="<?= $invoice['id'] ?>">
    
    <!-- Invoice Header -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0">Invoice Details</h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-md-6">
                    <div class="mb-3">
                        <label class="form-label">Client *</label>
                        <select name="client_id" class="form-select" required>
                            <option value="">Select Client</option>
                            <?php foreach ($clients as $client): ?>
                                <option value="<?= $client['Id'] ?>" <?= $client['Id'] == $invoice['client_id'] ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($client['CompanyName']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Invoice Number *</label>
                        <input type="text" name="invoice_number" class="form-control" value="<?= htmlspecialchars($invoice['invoice_number']) ?>" required>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Issue Date *</label>
                        <input type="date" name="issue_date" class="form-control" value="<?= $invoice['issue_date'] ?>" required>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="mb-3">
                        <label class="form-label">Due Date</label>
                        <input type="date" name="due_date" class="form-control" value="<?= $invoice['due_date'] ?>">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Currency *</label>
                        <select name="currency" class="form-select" required>
                            <option value="USD" <?= $invoice['currency'] == 'USD' ? 'selected' : '' ?>>USD</option>
                            <option value="SAR" <?= $invoice['currency'] == 'SAR' ? 'selected' : '' ?>>SAR</option>
                            <option value="EGP" <?= $invoice['currency'] == 'EGP' ? 'selected' : '' ?>>EGP</option>
                            <option value="EUR" <?= $invoice['currency'] == 'EUR' ? 'selected' : '' ?>>EUR</option>
                            <option value="GBP" <?= $invoice['currency'] == 'GBP' ? 'selected' : '' ?>>GBP</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Notes</label>
                        <textarea name="notes" class="form-control" rows="3"><?= htmlspecialchars($invoice['notes'] ?? '') ?></textarea>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Invoice Items -->
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Invoice Items</h5>
            <button type="button" class="btn btn-sm btn-primary" onclick="addInvoiceItem()">Add Item</button>
        </div>
        <div class="card-body">
            <div id="invoiceItems">
                <?php foreach ($invoice_items as $index => $item): ?>
                    <div class="invoice-item border rounded p-3 mb-3" data-index="<?= $index ?>">
                        <div class="row">
                            <div class="col-md-3">
                                <label class="form-label">Application *</label>
                                <select name="items[<?= $index ?>][application_id]" class="form-select" required>
                                    <option value="">Select Application</option>
                                    <?php foreach ($applications as $app): ?>
                                        <option value="<?= $app['id'] ?>" <?= $app['id'] == $item['application_id'] ? 'selected' : '' ?>>
                                            <?= htmlspecialchars($app['Application_Name']) ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label class="form-label">Description *</label>
                                <input type="text" name="items[<?= $index ?>][description]" class="form-control" value="<?= htmlspecialchars($item['description']) ?>" required>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Credit Units *</label>
                                <input type="number" name="items[<?= $index ?>][credit_units]" class="form-control" value="<?= $item['credit_units'] ?>" min="1" required>
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Expiry Date</label>
                                <input type="date" name="items[<?= $index ?>][expiry_date]" class="form-control" value="<?= $item['expiry_date'] ?>">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Unit Price *</label>
                                <input type="number" name="items[<?= $index ?>][unit_price]" class="form-control" value="<?= $item['unit_price'] ?>" min="0" step="0.01" required>
                            </div>
                        </div>
                        <div class="row mt-2">
                            <div class="col-md-2">
                                <label class="form-label">Quantity</label>
                                <input type="number" name="items[<?= $index ?>][quantity]" class="form-control" value="<?= $item['quantity'] ?>" min="1" step="0.01">
                            </div>
                            <div class="col-md-2">
                                <label class="form-label">Line Total</label>
                                <input type="text" class="form-control" value="<?= number_format($item['line_subtotal'], 2) ?>" readonly>
                            </div>
                            <div class="col-md-2">
                                <button type="button" class="btn btn-sm btn-danger mt-4" onclick="removeInvoiceItem(<?= $index ?>)">Remove</button>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
    
    <!-- Totals -->
    <div class="card mb-4">
        <div class="card-body">
            <div class="row">
                <div class="col-md-6 offset-md-6">
                    <div class="d-flex justify-content-between">
                        <strong>Subtotal:</strong>
                        <span id="subtotal"><?= number_format($invoice['subtotal'], 2) ?></span>
                    </div>
                    <div class="d-flex justify-content-between">
                        <strong>Total:</strong>
                        <span id="total"><?= number_format($invoice['total'], 2) ?></span>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Submit Buttons -->
    <div class="d-flex justify-content-between">
        <a href="?page=invoice_view&id=<?= $invoice['id'] ?>" class="btn btn-secondary">Cancel</a>
        <div>
            <button type="submit" class="btn btn-primary">Update Invoice</button>
            <button type="button" class="btn btn-success" onclick="issueInvoice()">Issue Invoice</button>
        </div>
    </div>
</form>

<script>
let itemIndex = <?= count($invoice_items) ?>;

function addInvoiceItem() {
    const container = document.getElementById('invoiceItems');
    const newItem = document.createElement('div');
    newItem.className = 'invoice-item border rounded p-3 mb-3';
    newItem.setAttribute('data-index', itemIndex);
    
    newItem.innerHTML = `
        <div class="row">
            <div class="col-md-3">
                <label class="form-label">Application *</label>
                <select name="items[${itemIndex}][application_id]" class="form-select" required>
                    <option value="">Select Application</option>
                    <?php foreach ($applications as $app): ?>
                        <option value="<?= $app['id'] ?>"><?= htmlspecialchars($app['Application_Name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-3">
                <label class="form-label">Description *</label>
                <input type="text" name="items[${itemIndex}][description]" class="form-control" required>
            </div>
            <div class="col-md-2">
                <label class="form-label">Credit Units *</label>
                <input type="number" name="items[${itemIndex}][credit_units]" class="form-control" min="1" required>
            </div>
            <div class="col-md-2">
                <label class="form-label">Expiry Date</label>
                <input type="date" name="items[${itemIndex}][expiry_date]" class="form-control">
            </div>
            <div class="col-md-2">
                <label class="form-label">Unit Price *</label>
                <input type="number" name="items[${itemIndex}][unit_price]" class="form-control" min="0" step="0.01" required>
            </div>
        </div>
        <div class="row mt-2">
            <div class="col-md-2">
                <label class="form-label">Quantity</label>
                <input type="number" name="items[${itemIndex}][quantity]" class="form-control" value="1" min="1" step="0.01">
            </div>
            <div class="col-md-2">
                <label class="form-label">Line Total</label>
                <input type="text" class="form-control" readonly>
            </div>
            <div class="col-md-2">
                <button type="button" class="btn btn-sm btn-danger mt-4" onclick="removeInvoiceItem(${itemIndex})">Remove</button>
            </div>
        </div>
    `;
    
    container.appendChild(newItem);
    itemIndex++;
}

function removeInvoiceItem(index) {
    const item = document.querySelector(`[data-index="${index}"]`);
    if (item) {
        item.remove();
        calculateTotals();
    }
}

function calculateTotals() {
    let subtotal = 0;
    const items = document.querySelectorAll('.invoice-item');
    
    items.forEach(item => {
        const unitPrice = parseFloat(item.querySelector('input[name*="[unit_price]"]').value) || 0;
        const quantity = parseFloat(item.querySelector('input[name*="[quantity]"]').value) || 1;
        const lineTotal = unitPrice * quantity;
        
        item.querySelector('input[readonly]').value = lineTotal.toFixed(2);
        subtotal += lineTotal;
    });
    
    document.getElementById('subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('total').textContent = subtotal.toFixed(2);
}

// Add event listeners for calculation
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('invoiceItems').addEventListener('input', function(e) {
        if (e.target.name.includes('[unit_price]') || e.target.name.includes('[quantity]')) {
            calculateTotals();
        }
    });
});

function issueInvoice() {
    if (confirm('Are you sure you want to issue this invoice? This action cannot be undone.')) {
        // Add a hidden field to indicate this should be issued
        const form = document.getElementById('invoiceEditForm');
        const issueField = document.createElement('input');
        issueField.type = 'hidden';
        issueField.name = 'issue_invoice';
        issueField.value = '1';
        form.appendChild(issueField);
        
        // Submit form via AJAX
        submitForm(form);
    }
}

function submitForm(form) {
    const formData = new FormData(form);
    
    fetch('api/invoices_update.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert(data.message);
            if (data.redirect_url) {
                window.location.href = data.redirect_url;
            }
        } else {
            alert('Error: ' + data.error);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred while processing the request.');
    });
}

// Update form submission to use AJAX
document.getElementById('invoiceEditForm').addEventListener('submit', function(e) {
    e.preventDefault();
    submitForm(this);
});
</script>
