<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get filter parameters
$client_id = $_GET['client_id'] ?? '';
$status = $_GET['status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build query
$where_conditions = [];
$params = [];

if ($client_id) {
    $where_conditions[] = "i.client_id = ?";
    $params[] = $client_id;
}

if ($status) {
    $where_conditions[] = "i.status = ?";
    $params[] = $status;
}

if ($date_from) {
    $where_conditions[] = "i.issue_date >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $where_conditions[] = "i.issue_date <= ?";
    $params[] = $date_to;
}

$where_clause = !empty($where_conditions) ? "WHERE " . implode(" AND ", $where_conditions) : "";

// Get invoices using existing $db1 connection
$invoices = [];
try {
    $query = "
        SELECT i.*, c.CompanyName as client_name, 
               COUNT(ii.id) as item_count,
               SUM(ii.line_subtotal) as calculated_total
        FROM invoices i
        LEFT JOIN tbl_clients c ON i.client_id = c.Id
        LEFT JOIN invoice_items ii ON i.id = ii.invoice_id
        {$where_clause}
        GROUP BY i.id
        ORDER BY i.issue_date DESC
    ";
    
    $stmt = $db1->prepare($query);
    $stmt->execute($params);
    $invoices = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Handle error silently for now
    $invoices = [];
}

// Get clients for filter
$clients = [];
try {
    $stmt = $db1->prepare("SELECT id, CompanyName FROM tbl_clients ORDER BY CompanyName");
    $stmt->execute();
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Handle error silently for now
    $clients = [];
}
?>

<!-- Invoices Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Invoices</h2>
    <a href="?page=invoice_create" class="btn btn-primary">Create Invoice</a>
</div>

<!-- Filters -->
<div class="card mb-4">
    <div class="card-body">
        <form method="GET" class="row g-3">
            <input type="hidden" name="page" value="invoices">
            <div class="col-md-3">
                <label class="form-label">Client</label>
                <select name="client_id" class="form-select">
                    <option value="">All Clients</option>
                    <?php foreach ($clients as $client): ?>
                        <option value="<?= $client['id'] ?>" <?= $client_id == $client['id'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($client['CompanyName']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">Status</label>
                <select name="status" class="form-select">
                    <option value="">All Statuses</option>
                    <option value="draft" <?= $status == 'draft' ? 'selected' : '' ?>>Draft</option>
                    <option value="issued" <?= $status == 'issued' ? 'selected' : '' ?>>Issued</option>
                    <option value="partially_paid" <?= $status == 'partially_paid' ? 'selected' : '' ?>>Partially Paid</option>
                    <option value="paid" <?= $status == 'paid' ? 'selected' : '' ?>>Paid</option>
                    <option value="void" <?= $status == 'void' ? 'selected' : '' ?>>Void</option>
                </select>
            </div>
            <div class="col-md-2">
                <label class="form-label">From Date</label>
                <input type="date" name="date_from" class="form-control" value="<?= $date_from ?>">
            </div>
            <div class="col-md-2">
                <label class="form-label">To Date</label>
                <input type="date" name="date_to" class="form-control" value="<?= $date_to ?>">
            </div>
            <div class="col-md-3 d-flex align-items-end">
                <button type="submit" class="btn btn-secondary me-2">Filter</button>
                <a href="?page=invoices" class="btn btn-outline-secondary">Clear</a>
            </div>
        </form>
    </div>
</div>

<!-- Invoices Table -->
<div class="card">
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Invoice #</th>
                        <th>Client</th>
                        <th>Issue Date</th>
                        <th>Due Date</th>
                        <th>Status</th>
                        <th>Currency</th>
                        <th>Total</th>
                        <th>Items</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($invoices)): ?>
                        <tr>
                            <td colspan="9" class="text-center">No invoices found</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($invoices as $invoice): ?>
                            <tr>
                                <td><?= htmlspecialchars($invoice['invoice_number']) ?></td>
                                <td><?= htmlspecialchars($invoice['client_name']) ?></td>
                                <td><?= $invoice['issue_date'] ?></td>
                                <td><?= $invoice['due_date'] ?: '-' ?></td>
                                <td>
                                    <span class="badge bg-<?= getStatusColor($invoice['status']) ?>">
                                        <?= ucfirst(str_replace('_', ' ', $invoice['status'])) ?>
                                    </span>
                                </td>
                                <td><?= htmlspecialchars($invoice['currency']) ?></td>
                                <td><?= number_format($invoice['total'], 2) ?></td>
                                <td><?= $invoice['item_count'] ?></td>
                                <td>
                                    <a href="?page=invoice_view&id=<?= $invoice['id'] ?>" class="btn btn-sm btn-info">View</a>
                                    <?php if ($invoice['status'] == 'draft'): ?>
                                        <a href="?page=invoice_edit&id=<?= $invoice['id'] ?>" class="btn btn-sm btn-warning">Edit</a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<?php
function getStatusColor($status) {
    switch ($status) {
        case 'draft': return 'secondary';
        case 'issued': return 'primary';
        case 'partially_paid': return 'warning';
        case 'paid': return 'success';
        case 'void': return 'danger';
        default: return 'secondary';
    }
}
?>
