<?php
// Security check
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: login.php");
    exit();
}

// CSRF Protection
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

require_once('includes/cache_handler.php');

try {
    $debug = filter_input(INPUT_GET, 'debug', FILTER_VALIDATE_INT) === 1;
    
    $cache = new DashboardCache();
    // Only use cache in non-debug mode
    $dashboardData = $debug ? null : $cache->getCachedData();
    
    if ($dashboardData === null) {
        $dashboardData = $cache->updateCache($db1, $debug);
    }
    
    if (!$dashboardData) {
        throw new Exception('Failed to retrieve dashboard data');
    }
    
    // Add debug information if requested
    if ($debug) {
        echo '<div class="alert alert-info">';
        echo '<h4>Cache Information</h4>';
        echo '<pre>';
        echo 'Cache Last Updated: ' . date('Y-m-d H:i:s', $dashboardData['timestamp']) . "\n";
        echo 'Cache Age: ' . (time() - $dashboardData['timestamp']) . " seconds\n";
        echo '</pre>';
        echo '</div>';
    }
    
    $topClients = $dashboardData['top_clients'] ?? [];
    $summary = $dashboardData['summary'] ?? [];
    ?>
    <!-- Loading overlay -->
    <div id="loading-overlay" style="display: none;">
        <div class="spinner-border text-primary" role="status">
            <span class="visually-hidden">Loading...</span>
        </div>
    </div>

    <div class="dashboard-container">
        <!-- Stats Overview -->
        <div class="row g-4 mb-5">
            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-primary-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-people-fill fs-4 text-primary"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Total Active Clients</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['total_clients'] ?? 107); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-primary" style="width: 70%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-success-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-file-text-fill fs-4 text-success"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Total Passports Processed</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['total_passports'] ?? 369947); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-success" style="width: 85%"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-lg-4 col-md-6">
                <div class="card stat-card border-0 shadow-sm">
                    <div class="card-body p-4">
                        <div class="d-flex align-items-center mb-3">
                            <div class="stat-icon bg-danger-subtle rounded-circle p-3 me-3">
                                <i class="bi bi-exclamation-triangle-fill fs-4 text-danger"></i>
                            </div>
                            <div>
                                <h6 class="card-subtitle text-muted mb-1">Low Credit Clients</h6>
                                <h2 class="card-title mb-0 fw-bold"><?php echo number_format($summary['low_credit'] ?? 36); ?></h2>
                            </div>
                        </div>
                        <div class="progress" style="height: 4px;">
                            <div class="progress-bar bg-danger" style="width: 30%"></div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Tab Navigation -->
        <ul class="nav nav-tabs mb-4" id="dashboardTabs" role="tablist">
            <li class="nav-item">
                <a class="nav-link active" id="overview-tab" data-bs-toggle="tab" href="#overview" role="tab">
                    Overview
                </a>
            </li>
            <li class="nav-item">
                <a class="nav-link" id="low-credit-tab" data-bs-toggle="tab" href="#low-credit" role="tab">
                    Low Credit Clients
                </a>
            </li>
        </ul>

        <!-- Tab Content -->
        <div class="tab-content">
            <!-- Overview Tab -->
            <div class="tab-pane fade show active" id="overview" role="tabpanel">
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Top Performing Clients</h5>
                        <button class="btn btn-sm btn-outline-primary" id="refreshData">
                            <i class="bi bi-arrow-clockwise"></i> Refresh
                        </button>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>City</th>
                                        <th>Total Passports</th>
                                        <th>Package</th>
                                        <th>Credit</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($topClients as $row) { ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($row['CompanyName']); ?></td>
                                            <td><?php echo htmlspecialchars($row['City']); ?></td>
                                            <td><?php echo number_format($row['total_passports']); ?></td>
                                            <td><?php echo number_format($row['package'] ?? 0); ?></td>
                                            <td><?php echo number_format($row['credit'] ?? 0); ?></td>
                                            <td>
                                                <a href="?page=edit_client&id=<?php echo $row['Id']; ?>" 
                                                   class="btn btn-sm btn-primary">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </a>
                                                <a href="download_passports.php?client_id=<?php echo $row['Id']; ?>" 
                                                   class="btn btn-sm btn-success">
                                                    <i class="bi bi-download"></i> Download Passports
                                                </a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Low Credit Clients Tab -->
            <div class="tab-pane fade" id="low-credit" role="tabpanel">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Clients with Low Credit</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-hover" id="lowCreditTable">
                                <thead>
                                    <tr>
                                        <th>Company Name</th>
                                        <th>Country</th>
                                        <th>Package Limit</th>
                                        <th>Used Passports</th>
                                        <th>Remaining Credit</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                <?php foreach ($dashboardData['low_credit_clients'] as $client): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($client['CompanyName']); ?></td>
                                        <td><?php echo htmlspecialchars($client['CountryName']); ?></td>
                                        <td><?php echo number_format($client['package_limit']); ?></td>
                                        <td><?php echo number_format($client['used_passports']); ?></td>
                                        <td>
                                            <span class="badge bg-<?php echo $client['remaining_credit'] <= 20 ? 'danger' : 'warning'; ?>">
                                                <?php echo number_format($client['remaining_credit']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <a href="index.php?page=edit_client&id=<?php echo $client['Id']; ?>" 
                                               class="btn btn-sm btn-primary">
                                                View Details
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Enhanced DataTables Initialization -->
        <script>
        $(document).ready(function() {
            // Initialize DataTables with improved configuration
            const lowCreditTable = $('#lowCreditTable').DataTable({
                order: [[4, 'asc']],
                pageLength: 25,
                responsive: true,
                language: {
                    search: "Search clients:",
                    lengthMenu: "Show _MENU_ clients per page",
                    info: "Showing _START_ to _END_ of _TOTAL_ clients"
                },
                dom: '<"d-flex justify-content-between align-items-center mb-3"lf>rtip'
            });

            // Updated refresh functionality
            $('#refreshData').on('click', function() {
                const $btn = $(this);
                const $icon = $btn.find('i');
                
                $btn.prop('disabled', true);
                $icon.addClass('rotate');
                $('#loading-overlay').show();

                // Trigger the cache update using the cron path
                $.post('cron/update_dashboard_cache.php', {
                    csrf_token: '<?php echo $_SESSION['csrf_token']; ?>'
                })
                .then(function(response) {
                    if (!response.success) {
                        throw new Error(response.message || 'Cache update failed');
                    }
                    // Reload the page to show updated data
                    location.reload();
                })
                .catch(function(error) {
                    showAlert(error.message || 'Failed to refresh data', 'danger');
                })
                .always(function() {
                    $btn.prop('disabled', false);
                    $icon.removeClass('rotate');
                    $('#loading-overlay').hide();
                });
            });

            // Helper function to show alerts
            function showAlert(message, type) {
                const alert = `
                    <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                        ${message}
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>`;
                $('.dashboard-container').prepend(alert);
            }
        });
        </script>

        <style>
        .rotate {
            animation: rotate 1s linear infinite;
        }
        @keyframes rotate {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        #loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 9999;
        }
        .dashboard-container {
            padding: 2rem;
            background-color: #f8f9fa;
        }

        .stat-card {
            transition: transform 0.2s ease-in-out;
            background: white;
        }

        .stat-card:hover {
            transform: translateY(-5px);
        }

        .stat-icon {
            width: 60px;
            height: 60px;
            display: flex;
            align-items: center;
            justify-content: center;
        }

        .progress {
            background-color: #e9ecef;
            border-radius: 2px;
        }

        /* Custom color utilities */
        .bg-primary-subtle {
            background-color: rgba(13, 110, 253, 0.1);
        }

        .bg-success-subtle {
            background-color: rgba(25, 135, 84, 0.1);
        }

        .bg-danger-subtle {
            background-color: rgba(220, 53, 69, 0.1);
        }

        /* Responsive font sizes */
        @media (max-width: 768px) {
            .card-title {
                font-size: 1.5rem;
            }
        }
        </style>
    <?php
} catch(Exception $e) {
    $errorMessage = $debug ? $e->getMessage() : 'An error occurred while fetching the dashboard data.';
    echo '<div class="alert alert-danger">
            <h4>Dashboard Error</h4>
            <p>' . htmlspecialchars($errorMessage) . '</p>
          </div>';
    error_log("Dashboard Error: " . $e->getMessage());
}
?> 
